/*
 * Copyright (c) 1999, 2000
 *	Politecnico di Torino.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the Politecnico
 * di Torino, and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */


#define UNICODE 1

#include <windows.h>
#include <windowsx.h>

#include "ntddndis.h"
#include "ntddpack.h"
#include "packet32.h"

TCHAR   szWindowTitle[] = TEXT("PACKET.DLL");

#if DBG
#define ODS(_x) OutputDebugString(TEXT(_x))
#else
#define ODS(_x)
#endif

//service handles
SC_HANDLE scmHandle = NULL;
SC_HANDLE srvHandle = NULL;
LPCTSTR PacketServiceName = TEXT("NPF");
LPCTSTR PacketServiceDesc = TEXT("Netgroup Packet Filter");


//---------------------------------------------------------------------------

BOOL APIENTRY DllMain (HANDLE DllHandle,DWORD Reason,LPVOID lpReserved)
{
    BOOLEAN Status=TRUE;

    ODS("Packet32: DllEntry\n");
    switch ( Reason )
    {
        case DLL_PROCESS_ATTACH:
            break;
        case DLL_PROCESS_DETACH:
			break;

		default:
            break;
    }

    return Status;
}

//---------------------------------------------------------------------------

WCHAR* SChar2WChar(char* nome)
{
	int i;
	WCHAR* TmpName;

	TmpName=(WCHAR*) malloc ((strlen(nome)+2)*sizeof(WCHAR));
	for (i=0;i<(signed)strlen(nome)+1; i++)
		TmpName[i]=nome[i];
	TmpName[i]=0;
	return TmpName;
}

//---------------------------------------------------------------------------
BOOLEAN PacketGetNetType (LPADAPTER AdapterObject,NetType *type)
{
    BOOLEAN    Status;
    ULONG      IoCtlBufferLength=(sizeof(PACKET_OID_DATA)+sizeof(ULONG)-1);
    PPACKET_OID_DATA  OidData;

    OidData=GlobalAllocPtr(GMEM_MOVEABLE | GMEM_ZEROINIT,IoCtlBufferLength);
    if (OidData == NULL) {
        return FALSE;
    }
	//get the link-layer type
    OidData->Oid = OID_GEN_MEDIA_IN_USE;
    OidData->Length = sizeof (ULONG);
    Status = PacketRequest(AdapterObject,FALSE,OidData);
    type->LinkType=*((UINT*)OidData->Data);

	//get the link-layer speed
    OidData->Oid = OID_GEN_LINK_SPEED;
    OidData->Length = sizeof (ULONG);
    Status = PacketRequest(AdapterObject,FALSE,OidData);
	type->LinkSpeed=*((UINT*)OidData->Data)*100;
    GlobalFreePtr (OidData);
    return Status;
}

//---------------------------------------------------------------------------

BOOLEAN PacketSetMaxLookaheadsize (LPADAPTER AdapterObject)
{
    BOOLEAN    Status;
    ULONG      IoCtlBufferLength=(sizeof(PACKET_OID_DATA)+sizeof(ULONG)-1);
    PPACKET_OID_DATA  OidData;

    OidData=GlobalAllocPtr(GMEM_MOVEABLE | GMEM_ZEROINIT,IoCtlBufferLength);
    if (OidData == NULL) {
        return FALSE;
    }

	//set the size of the lookahead buffer to the maximum available by the the NIC driver
    OidData->Oid=OID_GEN_MAXIMUM_LOOKAHEAD;
    OidData->Length=sizeof(ULONG);
    Status=PacketRequest(AdapterObject,FALSE,OidData);
    OidData->Oid=OID_GEN_CURRENT_LOOKAHEAD;
    Status=PacketRequest(AdapterObject,TRUE,OidData);
    GlobalFreePtr(OidData);
    return Status;
}

//---------------------------------------------------------------------------

BOOLEAN PacketSetReadEvt(LPADAPTER AdapterObject)
{
	DWORD BytesReturned;
	TCHAR EventName[39];

	// this tells the terminal service to retrieve the event from the global namespace
	wcsncpy(EventName,L"Global\\",sizeof(L"Global\\"));

	// retrieve the name of the shared event from the driver
	if(DeviceIoControl(AdapterObject->hFile,pBIOCEVNAME,NULL,0,EventName+7,13*sizeof(TCHAR),&BytesReturned,NULL)==FALSE) return FALSE;

	EventName[20]=0; // terminate the string

	// open the shared event
	AdapterObject->ReadEvent=CreateEvent(NULL,
										 TRUE,
										 FALSE,
										 EventName);

	// in NT4 "Global\" is not automatically ignored: try to use simply the event name
	if(GetLastError()!=ERROR_ALREADY_EXISTS){
		if(AdapterObject->ReadEvent != NULL)
			CloseHandle(AdapterObject->ReadEvent);
		
		// open the shared event
		AdapterObject->ReadEvent=CreateEvent(NULL,
			TRUE,
			FALSE,
			EventName+7);
	}	

	if(AdapterObject->ReadEvent==NULL || GetLastError()!=ERROR_ALREADY_EXISTS)
		return FALSE;

	AdapterObject->ReadTimeOut=0;

	return TRUE;

}


//---------------------------------------------------------------------------
//install the packet driver

BOOL PacketInstallDriver(SC_HANDLE ascmHandle,SC_HANDLE *srvHandle,TCHAR *driverPath)
{
	BOOL result = FALSE;
  
	if (GetFileAttributes(driverPath) != 0xffffffff) {
		*srvHandle = CreateService(ascmHandle, 
			PacketServiceName,
			PacketServiceDesc,
			SERVICE_ALL_ACCESS,
			SERVICE_KERNEL_DRIVER,
			SERVICE_DEMAND_START,
			SERVICE_ERROR_NORMAL,
			driverPath,
			NULL, NULL, NULL, NULL, NULL);
		if (*srvHandle == NULL) {
			if (GetLastError() == ERROR_SERVICE_EXISTS) {
				//packet.sys already existed
				result = TRUE;
			}
		}
		else {
			//Created service for packet.sys
			result = TRUE;
		}
	}
	if (result == TRUE) 
		if (*srvHandle != NULL)
			CloseServiceHandle(*srvHandle);
	return result;
	
}

//---------------------------------------------------------------------------

LPADAPTER PacketOpenAdapter(LPTSTR AdapterName)
{
    LPADAPTER  lpAdapter;
    BOOLEAN    Result;
	char	   *AdapterNameA;
	WCHAR	   *AdapterNameU;
	DWORD	error;
	SC_HANDLE svcHandle = NULL;
	TCHAR driverPath[512];
	TCHAR WinPath[256];
	
	ODS("Packet32: PacketOpenAdapter\n");

	scmHandle = OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS);
	
	*driverPath = 0;
	GetCurrentDirectory(512, driverPath);
	wsprintf(driverPath + wcslen(driverPath), 
		TEXT("\\packet.sys"));

	if (PacketInstallDriver(scmHandle,&svcHandle,driverPath)) {
		srvHandle = OpenService(scmHandle,PacketServiceName,SERVICE_START);
		if (srvHandle != NULL){
 			if (StartService(srvHandle, 0, NULL)==0){ 
				error = GetLastError();
				if(error!=1056){
					SetLastError(error);
					if (scmHandle != NULL) CloseServiceHandle(scmHandle);
					return NULL;
				}
			}
		}
		else return NULL; 
	}
	else{
		if( GetWindowsDirectory(WinPath, sizeof(WinPath)/sizeof(TCHAR)) == 0) return FALSE;
		wsprintf(driverPath,
			TEXT("%s\\system32\\drivers\\packet.sys"), 
			WinPath);
		
		if (PacketInstallDriver(scmHandle,&svcHandle,driverPath)) {
			srvHandle = OpenService(scmHandle,PacketServiceName,SERVICE_START);
			if (srvHandle != NULL){
				if (StartService(srvHandle, 0, NULL)==0){ 
					error = GetLastError();
					if(error!=1056){
						SetLastError(error);
						if (scmHandle != NULL) CloseServiceHandle(scmHandle);
						return NULL;
					}
				}
			}
			else return NULL; 
		}
	}

    if (scmHandle != NULL) CloseServiceHandle(scmHandle);

	AdapterNameA=(char*)AdapterName;
	if(AdapterNameA[1]!=0){ //ASCII
		AdapterNameU=SChar2WChar(AdapterNameA);
		AdapterName=AdapterNameU;
	} else {			//Unicode
		AdapterNameU=NULL;
	}
	
	lpAdapter=(LPADAPTER)GlobalAllocPtr(GMEM_MOVEABLE | GMEM_ZEROINIT,sizeof(ADAPTER));
	if (lpAdapter==NULL)
	{
		ODS("Packet32: PacketOpenAdapter GlobalAlloc Failed\n");
		error=GetLastError();
		if (AdapterNameU != NULL) free(AdapterNameU);
		//set the error to the one on which we failed
		SetLastError(error);
		return NULL;
	}
	lpAdapter->NumWrites=1;

	wsprintf(lpAdapter->SymbolicLink,TEXT("\\\\.\\%s%s"),DOSNAMEPREFIX,&AdapterName[8]);
	
	//try if it is possible to open the adapter immediately
	lpAdapter->hFile=CreateFile(lpAdapter->SymbolicLink,GENERIC_WRITE | GENERIC_READ,
		0,NULL,OPEN_EXISTING,0,0);
	
	if (lpAdapter->hFile != INVALID_HANDLE_VALUE) {

		if(PacketSetReadEvt(lpAdapter)==FALSE){
			error=GetLastError();
			ODS("Packet32: Unable to open the read event\n");
			if (AdapterNameU != NULL)
				free(AdapterNameU);
			GlobalFreePtr(lpAdapter);
			//set the error to the one on which we failed
			SetLastError(error);
			return NULL;
		}		
		
		if (AdapterNameU != NULL)
		    free(AdapterNameU);
		return lpAdapter;
	}
	//this is probably the first request on the packet driver. 
	//We must create the dos device and set the access rights on it
	else{
		Result=DefineDosDevice(DDD_RAW_TARGET_PATH,&lpAdapter->SymbolicLink[4],AdapterName);
		if (Result)
		{

			lpAdapter->hFile=CreateFile(lpAdapter->SymbolicLink,GENERIC_WRITE | GENERIC_READ,
				0,NULL,OPEN_EXISTING,0,0);
			if (lpAdapter->hFile != INVALID_HANDLE_VALUE)
			{		
				
				if(PacketSetReadEvt(lpAdapter)==FALSE){
					error=GetLastError();
					ODS("Packet32: Unable to open the read event\n");
					if (AdapterNameU != NULL)
						free(AdapterNameU);
					GlobalFreePtr(lpAdapter);
					//set the error to the one on which we failed
					SetLastError(error);
					return NULL;					
				}

				PacketSetMaxLookaheadsize(lpAdapter);
				if (AdapterNameU != NULL)
				    free(AdapterNameU);
				return lpAdapter;
			}
		}
	}

	error=GetLastError();
	ODS("Packet32: PacketOpenAdapter Could not open adapter\n");
	if (AdapterNameU != NULL)
	    free(AdapterNameU);
	GlobalFreePtr(lpAdapter);
	//set the error to the one on which we failed
	SetLastError(error);
	return NULL;

}

//---------------------------------------------------------------------------

VOID PacketCloseAdapter(LPADAPTER lpAdapter)
{
    ODS("Packet32: PacketCloseAdapter\n");
    CloseHandle(lpAdapter->hFile);
	SetEvent(lpAdapter->ReadEvent);
    CloseHandle(lpAdapter->ReadEvent);
    GlobalFreePtr(lpAdapter);
}

//---------------------------------------------------------------------------

LPPACKET PacketAllocatePacket(void)
{

    LPPACKET    lpPacket;
    lpPacket=(LPPACKET)GlobalAllocPtr(GMEM_MOVEABLE | GMEM_ZEROINIT,sizeof(PACKET));
    if (lpPacket==NULL)
    {
        ODS("Packet32: PacketAllocatePacket: GlobalAlloc Failed\n");
        return NULL;
    }
    return lpPacket;
}

//---------------------------------------------------------------------------

VOID PacketFreePacket(LPPACKET lpPacket)

{
    GlobalFreePtr(lpPacket);
}

//---------------------------------------------------------------------------

VOID PacketInitPacket(LPPACKET lpPacket,PVOID Buffer,UINT Length)

{
    lpPacket->Buffer = Buffer;
    lpPacket->Length = Length;
	lpPacket->ulBytesReceived = 0;
	lpPacket->bIoComplete = FALSE;
}

//---------------------------------------------------------------------------

BOOLEAN PacketSendPacket(LPADAPTER AdapterObject,LPPACKET lpPacket,BOOLEAN Sync)

{
    DWORD        BytesTransfered;
    
    return WriteFile(AdapterObject->hFile,lpPacket->Buffer,lpPacket->Length,&BytesTransfered,NULL);
}


//---------------------------------------------------------------------------

BOOLEAN PacketReceivePacket(LPADAPTER AdapterObject,LPPACKET lpPacket,BOOLEAN Sync)
{
	if((int)AdapterObject->ReadTimeOut != -1)
		WaitForSingleObject(AdapterObject->ReadEvent, (AdapterObject->ReadTimeOut==0)?INFINITE:AdapterObject->ReadTimeOut);

    return ReadFile(AdapterObject->hFile, lpPacket->Buffer, lpPacket->Length, &lpPacket->ulBytesReceived,NULL);
}


//---------------------------------------------------------------------------
/* Function to set the working mode of the driver
mode	working mode
*/

BOOLEAN PacketSetMode(LPADAPTER AdapterObject,int mode)
{
	int BytesReturned;

    return DeviceIoControl(AdapterObject->hFile,pBIOCSMODE,&mode,4,NULL,0,&BytesReturned,NULL);
}

//---------------------------------------------------------------------------
/* Function to set the the number of times a write will be repeated*/

BOOLEAN PacketSetNumWrites(LPADAPTER AdapterObject,int nwrites)
{
	int BytesReturned;
    return DeviceIoControl(AdapterObject->hFile,pBIOCSWRITEREP,&nwrites,4,NULL,0,&BytesReturned,NULL);
}

//---------------------------------------------------------------------------
/* This function is used to set the read timeout
dim		value of timeout(milliseconds). 0 means infinite.
*/

BOOLEAN PacketSetReadTimeout(LPADAPTER AdapterObject,int timeout)
{
	int BytesReturned;
	int DriverTimeOut=-1;

	if(timeout==0) AdapterObject->ReadTimeOut= INFINITE;
	else if(timeout==-1) AdapterObject->ReadTimeOut= 0;
	else AdapterObject->ReadTimeOut=timeout;

    return DeviceIoControl(AdapterObject->hFile,pBIOCSRTIMEOUT,&DriverTimeOut,4,NULL,0,&BytesReturned,NULL);
}

//---------------------------------------------------------------------------
/* This function allows to set the dimension of the packet buffer in the driver
parameters:
dim		dimension of the buffer (kilobytes)
*/

BOOLEAN PacketSetBuff(LPADAPTER AdapterObject,int dim)
{
	int BytesReturned;
    return DeviceIoControl(AdapterObject->hFile,pBIOCSETBUFFERSIZE,&dim,4,NULL,0,&BytesReturned,NULL);
}

//---------------------------------------------------------------------------
/* Function to set the minimum amount of bytes that will be copied by the driver*/

BOOLEAN PacketSetMinToCopy(LPADAPTER AdapterObject,int nbytes)
{
	int BytesReturned;
    return DeviceIoControl(AdapterObject->hFile,pBIOCSMINTOCOPY,&nbytes,4,NULL,0,&BytesReturned,NULL);
}

//---------------------------------------------------------------------------
/* Function to set a bpf filter in the driver
parameters:
fp		the pointer to the beginning of the filtering program
*/

BOOLEAN PacketSetBpf(LPADAPTER AdapterObject,struct bpf_program *fp)
{
	int BytesReturned;
    return DeviceIoControl(AdapterObject->hFile,pBIOCSETF,(char*)fp->bf_insns,fp->bf_len*sizeof(struct bpf_insn),NULL,0,&BytesReturned,NULL);
}

//---------------------------------------------------------------------------
/* Function to get the number of packet received and dropped from the driver
parameters:
s		structure containig 2 int values that will be filled by the driver
*/

BOOLEAN PacketGetStats(LPADAPTER AdapterObject,struct bpf_stat *s)
{
	int BytesReturned;
	return DeviceIoControl(AdapterObject->hFile,pBIOCGSTATS,NULL,0,s,sizeof(struct bpf_stat),&BytesReturned,NULL);
}

//---------------------------------------------------------------------------

BOOLEAN PacketResetAdapter(LPADAPTER  AdapterObject)

{
    UINT       BytesReturned;

    DeviceIoControl(AdapterObject->hFile,(DWORD)IOCTL_PROTOCOL_RESET,NULL,0,NULL,0,&BytesReturned,NULL);
    return TRUE;
}

//---------------------------------------------------------------------------

BOOLEAN PacketRequest(LPADAPTER  AdapterObject,BOOLEAN Set,PPACKET_OID_DATA  OidData)
{
    UINT       BytesReturned;
    BOOLEAN    Result;

    Result=DeviceIoControl(AdapterObject->hFile,(DWORD) Set ? pBIOCSETOID : pBIOCQUERYOID,
                           OidData,sizeof(PACKET_OID_DATA)-1+OidData->Length,OidData,
                           sizeof(PACKET_OID_DATA)-1+OidData->Length,&BytesReturned,NULL);
    return Result;
}

//---------------------------------------------------------------------------

BOOLEAN PacketSetHwFilter(LPADAPTER  AdapterObject,ULONG Filter)
{
    BOOLEAN    Status;
    ULONG      IoCtlBufferLength=(sizeof(PACKET_OID_DATA)+sizeof(ULONG)-1);
    PPACKET_OID_DATA  OidData;

    OidData=GlobalAllocPtr(GMEM_MOVEABLE | GMEM_ZEROINIT,IoCtlBufferLength);
    if (OidData == NULL) {
        return FALSE;
    }
    OidData->Oid=OID_GEN_CURRENT_PACKET_FILTER;
    OidData->Length=sizeof(ULONG);
    *((PULONG)OidData->Data)=Filter;
    Status=PacketRequest(AdapterObject,TRUE,OidData);
    GlobalFreePtr(OidData);
    return Status;
}

//---------------------------------------------------------------------------

BOOLEAN PacketGetAdapterNames(PTSTR pStr,PULONG  BufferSize)
{
    HKEY       LinkageKey,AdapKey;
	UINT	   RegKeySize=0;
    LONG       Status;
	ULONG	   Result;
	PTSTR      BpStr;
	char       *TTpStr,*DpStr,*DescBuf;
	LPADAPTER  adapter;
    PPACKET_OID_DATA  OidData;
	int		   i=0,k,rewind,dim;
	TCHAR	   AdapName[256];

    OidData=GlobalAllocPtr(GMEM_MOVEABLE | GMEM_ZEROINIT,512);
    if (OidData == NULL) {
        return FALSE;
    }


    Status=RegOpenKeyEx(HKEY_LOCAL_MACHINE,
		                TEXT("SYSTEM\\CurrentControlSet\\Control\\Class\\{4D36E972-E325-11CE-BFC1-08002BE10318}"),
						0,
						KEY_READ,
						&AdapKey);

	// Get the size to allocate for the original device names
	while((Result=RegEnumKey(AdapKey,i,AdapName,sizeof(AdapName)/2))==ERROR_SUCCESS)
	{
		Status=RegOpenKeyEx(AdapKey,AdapName,0,KEY_READ,&LinkageKey);
		Status=RegOpenKeyEx(LinkageKey,L"Linkage",0,KEY_READ,&LinkageKey);
        Status=RegQueryValueEx(LinkageKey,L"Export",NULL,NULL,NULL,&dim);
		i++;
		if(Status!=ERROR_SUCCESS) continue;
		RegKeySize+=dim;
	}
	
	// Allocate the memory for the original device names
	BpStr=GlobalAllocPtr(GMEM_MOVEABLE | GMEM_ZEROINIT,RegKeySize+2);
	if (BpStr == NULL || RegKeySize > *BufferSize) {
		GlobalFreePtr(OidData);
		return FALSE;
	}

	k=0;
	i=0;

	// Copy the names to the buffer
	while((Result=RegEnumKey(AdapKey,i,AdapName,sizeof(AdapName)/2))==ERROR_SUCCESS)
	{
		WCHAR UpperBindStr[64];

		i++;

		Status=RegOpenKeyEx(AdapKey,AdapName,0,KEY_READ,&LinkageKey);
		Status=RegOpenKeyEx(LinkageKey,L"Linkage",0,KEY_READ,&LinkageKey);

		dim=sizeof(UpperBindStr);
        Status=RegQueryValueEx(LinkageKey,L"UpperBind",NULL,NULL,(PUCHAR)UpperBindStr,&dim);

		if( Status!=ERROR_SUCCESS || wcscmp(UpperBindStr,L"NdisWan")==0 ) continue;

		dim=RegKeySize-k;
        Status=RegQueryValueEx(LinkageKey,L"Export",NULL,NULL,(LPBYTE)BpStr+k,&dim);
		if(Status!=ERROR_SUCCESS) continue;
		k+=dim-2;
	}

	CloseHandle(AdapKey);
	
	if (k != 0){
		
		DescBuf=GlobalAllocPtr(GMEM_MOVEABLE | GMEM_ZEROINIT, 4096);
		if (DescBuf == NULL) {
			GlobalFreePtr (BpStr);
		    GlobalFreePtr(OidData);
			return FALSE;
		}
		DpStr=DescBuf;
				
		for(i=0,k=0;BpStr[i]!=0 || BpStr[i+1]!=0;){
			
			if(k+wcslen(BpStr+i)+30 > *BufferSize){
				// Input buffer too small
			    GlobalFreePtr(OidData);
				GlobalFreePtr (BpStr);
				GlobalFreePtr (DescBuf);
				return FALSE;
			}

			// Create the device name
			rewind=k;
			memcpy(pStr+k,BpStr+i,16);
			memcpy(pStr+k+8,TEXT("Packet_"),14);
			i+=8;
			k+=15;
			while(BpStr[i-1]!=0){
				pStr[k++]=BpStr[i++];
			}

			// Open the adapter
			adapter=PacketOpenAdapter(pStr+rewind);
			if(adapter==NULL){
				k=rewind;
				continue;
			}

			// Retrieve the description
			OidData->Oid = OID_GEN_VENDOR_DESCRIPTION;
			OidData->Length = 256;
			ZeroMemory(OidData->Data,256);
			Status = PacketRequest(adapter,FALSE,OidData);
			if(Status==0 || ((char*)OidData->Data)[0]==0){
				k=rewind;
				continue;
			}

			// Copy the description
			TTpStr=(char*)(OidData->Data);
			while(*TTpStr!=0){
				*DpStr++=*TTpStr++;
			}
			*DpStr++=*TTpStr++;
			
			// Close the adapter
			PacketCloseAdapter(adapter);
			
		}
		*DpStr=0;

		pStr[k++]=0;
		pStr[k]=0;

		if((ULONG)(DpStr-DescBuf+k) < *BufferSize)
			memcpy(pStr+k,DescBuf,DpStr-DescBuf);
		else{
		    GlobalFreePtr(OidData);
			GlobalFreePtr (BpStr);
			GlobalFreePtr (DescBuf);
			return FALSE;
		}

	    GlobalFreePtr(OidData);
		GlobalFreePtr (BpStr);
		GlobalFreePtr (DescBuf);
		return TRUE;
	}
	else{
	    DWORD      RegType;

		GlobalFreePtr (BpStr);

		Status=RegOpenKeyEx(HKEY_LOCAL_MACHINE,TEXT("SYSTEM\\CurrentControlSet\\Services\\Tcpip\\Linkage"),0,KEY_READ,&LinkageKey);
		if (Status == ERROR_SUCCESS)
		{
			// Retrieve the length of the key
			Status=RegQueryValueEx(LinkageKey,TEXT("bind"),NULL,&RegType,NULL,&RegKeySize);
			// Allocate the buffer
			BpStr=GlobalAllocPtr(GMEM_MOVEABLE | GMEM_ZEROINIT,RegKeySize+2);
			if (BpStr == NULL || RegKeySize > *BufferSize) {
				GlobalFreePtr(OidData);
				return FALSE;
			}
			Status=RegQueryValueEx(LinkageKey,TEXT("bind"),NULL,&RegType,(LPBYTE)BpStr,&RegKeySize);
			RegCloseKey(LinkageKey);
		}
		
		if (Status==ERROR_SUCCESS){
			
			DescBuf=GlobalAllocPtr(GMEM_MOVEABLE | GMEM_ZEROINIT, 4096);
			if (DescBuf == NULL) {
				GlobalFreePtr (BpStr);
				GlobalFreePtr(OidData);
				return FALSE;
			}
			DpStr=DescBuf;
			
			for(i=0,k=0;BpStr[i]!=0 || BpStr[i+1]!=0;){
				
				if(k+wcslen(BpStr+i)+30 > *BufferSize){
					// Input buffer too small
					GlobalFreePtr(OidData);
					GlobalFreePtr (BpStr);
					GlobalFreePtr (DescBuf);
					return FALSE;
				}
				
				// Create the device name
				rewind=k;
				memcpy(pStr+k,BpStr+i,16);
				memcpy(pStr+k+8,TEXT("Packet_"),14);
				i+=8;
				k+=15;
				while(BpStr[i-1]!=0){
					pStr[k++]=BpStr[i++];
				}
				
				// Open the adapter
				adapter=PacketOpenAdapter(pStr+rewind);
				if(adapter==NULL){
					k=rewind;
					continue;
				}
				
				// Retrieve the description
				OidData->Oid = OID_GEN_VENDOR_DESCRIPTION;
				OidData->Length = 256;
				Status = PacketRequest(adapter,FALSE,OidData);
				if(Status==0 || ((char*)OidData->Data)[0]==0){
					k=rewind;
					continue;
				}
				
				// Copy the description
				TTpStr=(char*)(OidData->Data);
				while(*TTpStr!=0){
					*DpStr++=*TTpStr++;
				}
				*DpStr++=*TTpStr++;
				
				// Close the adapter
				PacketCloseAdapter(adapter);
				
			}
			*DpStr=0;
			
			pStr[k++]=0;
			pStr[k]=0;
			
			if((ULONG)(DpStr-DescBuf+k) < *BufferSize)
				memcpy(pStr+k,DescBuf,DpStr-DescBuf);
			else{
				GlobalFreePtr(OidData);
				GlobalFreePtr (BpStr);
				GlobalFreePtr (DescBuf);
				return FALSE;
			}
			
			GlobalFreePtr(OidData);
			GlobalFreePtr (BpStr);
			GlobalFreePtr (DescBuf);
			return TRUE;
		}
		else{
			MessageBox(NULL,TEXT("Can not find TCP/IP bindings.\nIn order to run the packet capture driver you must install TCP/IP."),szWindowTitle,MB_OK);
			return FALSE;
		}
	}
}

//---------------------------------------------------------------------------
/* Convert a Unicode dotted-quad to a 32-bit IP address.
   Doesn't check to make sure it's valid. */

ULONG inet_addrU(const WCHAR *cp)
{
	ULONG val, part;
	WCHAR c;
	int i;

	val = 0;
	for (i = 0; i < 4; i++) {
		part = 0;
		while ((c = *cp++) != '\0' && c != '.') {
			if (c < '0' || c > '9')
				return -1;
			part = part*10 + (c - '0');
		}
		if (part > 255)
			return -1;	
		val = (val << 8) | part;
		if (i == 3) {
			if (c != '\0')
				return -1;	// extra gunk at end of string 
		} else {
			if (c == '\0')
				return -1;	// string ends early 
		}
	}
	return val;
}

//---------------------------------------------------------------------------

BOOLEAN PacketGetNetInfo(LPTSTR AdapterName, PULONG netp, PULONG maskp)
{
	char	*AdapterNameA;
	WCHAR	*AdapterNameU;
	WCHAR	*ifname;
	HKEY	SystemKey;
	HKEY	ControlSetKey;
	HKEY	ServicesKey;
	HKEY	InterfaceKey;
	HKEY	ParametersKey;
	HKEY	TcpIpKey;
	LONG	status,statusm;
	WCHAR	String[1024+1];
	DWORD	RegType;
	ULONG	BufLen;
	DWORD	DHCPEnabled;

	AdapterNameA = (char*)AdapterName;
	if(AdapterNameA[1] != 0) {	//ASCII
		AdapterNameU = SChar2WChar(AdapterNameA);
		AdapterName = AdapterNameU;
	} else {				//Unicode
		AdapterNameU = NULL;
	}
	ifname = wcsrchr(AdapterName, '\\');
	if (ifname == NULL)
		ifname = AdapterName;
	else
		ifname++;
	if (wcsncmp(ifname, L"Packet_", 7) == 0)
		ifname += 7;
	status = RegOpenKeyEx(HKEY_LOCAL_MACHINE,TEXT("SYSTEM"),0,KEY_READ,&SystemKey);
	if (status != ERROR_SUCCESS)
		goto fail;
	status = RegOpenKeyEx(SystemKey,TEXT("CurrentControlSet"),0,KEY_READ,&ControlSetKey);
	if (status != ERROR_SUCCESS) {
		RegCloseKey(SystemKey);
		goto fail;
	}
	RegCloseKey(SystemKey);
	status = RegOpenKeyEx(ControlSetKey,TEXT("Services"),0,KEY_READ,&ServicesKey);
	if (status != ERROR_SUCCESS) {
		RegCloseKey(ControlSetKey);
		goto fail;
	}
	RegCloseKey(ControlSetKey);
	status = RegOpenKeyEx(ServicesKey,ifname,0,KEY_READ,&InterfaceKey);
	if (status != ERROR_SUCCESS) {
		RegCloseKey(ServicesKey);
		goto fail;
	}
	RegCloseKey(ServicesKey);
	status = RegOpenKeyEx(InterfaceKey,TEXT("Parameters"),0,KEY_READ,&ParametersKey);
	if (status != ERROR_SUCCESS) {
		RegCloseKey(InterfaceKey);
		goto fail;
	}
	RegCloseKey(InterfaceKey);
	status = RegOpenKeyEx(ParametersKey,TEXT("TcpIp"),0,KEY_READ,&TcpIpKey);
	if (status != ERROR_SUCCESS) {
		RegCloseKey(ParametersKey);
		goto fail;
	}
	RegCloseKey(ParametersKey);
	BufLen = sizeof String;
	status = RegQueryValueEx(TcpIpKey,TEXT("IPAddress"),NULL,&RegType,(LPBYTE)String,&BufLen);
	if (status != ERROR_SUCCESS) {
		RegCloseKey(TcpIpKey);
		goto fail;
	}
	*netp = inet_addrU(String);
	BufLen = 4;
	
	/* See if DHCP is used by this system */
	status=RegQueryValueEx(TcpIpKey,TEXT("EnableDHCP"),NULL,&RegType,(LPBYTE)&DHCPEnabled,&BufLen);
	if (status != ERROR_SUCCESS) {
		statusm=RegQueryValueEx(TcpIpKey,TEXT("SubnetMask"),NULL,&RegType,
			(LPBYTE)String,&BufLen);
		
		if (statusm != ERROR_SUCCESS) {
			RegCloseKey(TcpIpKey);
			goto fail;
		}
		
		*maskp = inet_addrU(String);
		if (AdapterNameU != NULL)
			free(AdapterNameU);
		return TRUE;
		
	}
	
	status=1;
	BufLen = sizeof String;
	
	/* Retrieve the netmask */
	if(DHCPEnabled){
		status=RegQueryValueEx(TcpIpKey,TEXT("DHCPSubnetMask"),NULL,&RegType,
			(LPBYTE)String,&BufLen);
	}
	else{
			status=RegQueryValueEx(TcpIpKey,TEXT("SubnetMask"),NULL,&RegType,
			(LPBYTE)String,&BufLen);
	}
	
	if (status != ERROR_SUCCESS) {
		RegCloseKey(TcpIpKey);
		goto fail;
	}
	
	
	*maskp = inet_addrU(String);
	if (AdapterNameU != NULL)
		free(AdapterNameU);
	return TRUE;

fail:
	if (AdapterNameU != NULL)
		free(AdapterNameU);
    return FALSE;
}


