/* Read functions.

Copyright (C) 1999 Politecnico di Torino

This file is part of the NDIS Packet capture driver.

The GNU C Library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

The GNU C Library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with the GNU C Library; see the file COPYING.LIB.  If
not, write to the Free Software Foundation, Inc., 675 Mass Ave,
Cambridge, MA 02139, USA.  */


#include <stdarg.h>
#include <ntddk.h>
#include <ntiologc.h>
#include <ndis.h>
#include "debug.h"
#include "packet.h"

VOID
PacketCancelRoutine (
    IN PDEVICE_OBJECT   DeviceObject,
    IN PIRP             Irp
);


//-------------------------------------------------------------------
void PacketMoveMem(PVOID Destination,
				   PVOID Source, 
				   ULONG Length,
				   UINT	 *Bhead)
{
/*
Routine Description:
	Move a portion of the circular buffer updating the head every 1024 bytes
*/

ULONG WordLength;
UINT n,i,NBlocks;

	WordLength=Length>>2;
	NBlocks=WordLength>>8;
	
	for(n=0;n<NBlocks;n++){
		for(i=0;i<256;i++){
			*((PULONG)Destination)++=*((PULONG)Source)++;
		}
	*Bhead+=1024;
	}

	n=WordLength-(NBlocks<<8);
	for(i=0;i<n;i++){
		*((PULONG)Destination)++=*((PULONG)Source)++;
	}
	*Bhead+=n<<2;
	
	n=Length-(WordLength<<2);
	for(i=0;i<n;i++){
		*((PUCHAR)Destination)++=*((PUCHAR)Source)++;
	}
	*Bhead+=n;
}


//-------------------------------------------------------------------
NTSTATUS PacketRead(IN PDEVICE_OBJECT DeviceObject,IN PIRP Irp)
/*
Routine Description:
    This is the dispatch routine for read requests.
Arguments:
    DeviceObject - Pointer to the device object.
    Irp - Pointer to the request packet.
Return Value:
    Status is returned.
*/

{
    POPEN_INSTANCE      Open;
    PIO_STACK_LOCATION  IrpSp;
    PNDIS_PACKET        pPacket;
    PMDL                pMdl;
    NDIS_STATUS         Status;
    PLIST_ENTRY         PacketListEntry;
    PUCHAR				packp;//buffer that maps the application memory
	UINT				i;
	ULONG				Input_Buffer_Length;
	UINT				Thead;
	UINT				Ttail;
	UINT				TLastByte;
	KIRQL				OldIrql;
	PUCHAR				CurrBuff;
	UINT				cplen;
	UINT				CpStart;

	IF_LOUD(DbgPrint("Packet: Read\n");)
    IrpSp = IoGetCurrentIrpStackLocation(Irp);
    Open=IrpSp->FileObject->FsContext;
    //  See if the buffer is atleast big enough to hold the
    //  ethernet header
    if (IrpSp->Parameters.Read.Length < ETHERNET_HEADER_LENGTH)
    {
        Irp->IoStatus.Status = STATUS_BUFFER_TOO_SMALL;
		Irp->IoStatus.Information = 0;
	    IoCompleteRequest(Irp, IO_NO_INCREMENT);
        return STATUS_BUFFER_TOO_SMALL;
    }

	//See if there are packets in the buffer

	Thead=Open->Bhead;
	Ttail=Open->Btail;
	TLastByte=Open->BLastByte;

	if (Thead == Ttail)
		{
		//there aren't buffered packet: the application must wait

		    //  Allocate an MDL to map the portion of the buffer following the
			//  header
			pMdl=IoAllocateMdl(MmGetMdlVirtualAddress(Irp->MdlAddress),
		                   MmGetMdlByteCount(Irp->MdlAddress),FALSE,FALSE,NULL);
			if (pMdl == NULL)
			{
				IF_LOUD(DbgPrint("Packet: Read-Failed to allocate Mdl\n");)
				Irp->IoStatus.Status = STATUS_INSUFFICIENT_RESOURCES;
				Irp->IoStatus.Information = 0;
			    IoCompleteRequest(Irp, IO_NO_INCREMENT);
				return STATUS_INSUFFICIENT_RESOURCES;
			}
			//  Build the mdl to point to the the portion of the buffer followin
			//  the bpf data and the mac header
			IoBuildPartialMdl(Irp->MdlAddress,pMdl,((PUCHAR)MmGetMdlVirtualAddress(Irp->MdlAddress))+ETHERNET_HEADER_LENGTH+sizeof(struct bpf_hdr),0);
			//  Clear the next link in the new MDL
			pMdl->Next=NULL;
			//  Try to get a packet from our list of free ones
			NdisAllocatePacket(&Status,&pPacket,Open->PacketPool);
			if (Status != NDIS_STATUS_SUCCESS)
			{
			    IF_LOUD(DbgPrint("Packet: Read- No free packets\n");)
			    IoFreeMdl(pMdl);
			    Irp->IoStatus.Status = STATUS_INSUFFICIENT_RESOURCES;
				Irp->IoStatus.Information = 0;
			    IoCompleteRequest(Irp, IO_NO_INCREMENT);
				return STATUS_INSUFFICIENT_RESOURCES;
			}
			//  Get a pointer to the packet itself
			RESERVED(pPacket)->Irp=Irp;
			RESERVED(pPacket)->pMdl=pMdl;
	
		    IoMarkIrpPending(Irp);

		    //  Attach our new MDL to the packet
		    NdisChainBufferAtFront(pPacket,pMdl);
		    //  Put this packet in a list of pending reads.
		    //  The receive indication handler will attemp to remove packets
		    //  from this list for use in transfer data calls
		    ExInterlockedInsertTailList(&Open->RcvList,&RESERVED(pPacket)->ListElement,&Open->RcvQSpinLock);

			IoSetCancelRoutine(Irp, PacketCancelRoutine);

			return(STATUS_PENDING);
		}

		//there is at least a buffered packet
		//the read call is completed

		Input_Buffer_Length=IrpSp->Parameters.Read.Length;
		packp=(PUCHAR)MmGetMdlVirtualAddress(Irp->MdlAddress);
		i=0;
		//get the address of the buffer
		CurrBuff=Open->Buffer;
		
		//fill the application buffer

		//first of all see if it we can copy all the buffer in one time
		if(Ttail>Thead){
			if((Ttail-Thead)<Input_Buffer_Length){
				PacketMoveMem(packp,CurrBuff+Thead,Ttail-Thead,&(Open->Bhead));
				Irp->IoStatus.Information = Ttail-Thead;
				Irp->IoStatus.Status = STATUS_SUCCESS;
				IoCompleteRequest(Irp, IO_NO_INCREMENT);
				return(STATUS_SUCCESS);
			}
		}
		else if((TLastByte-Thead)<Input_Buffer_Length){
				PacketMoveMem(packp,CurrBuff+Thead,TLastByte-Thead,&(Open->Bhead));
				Open->BLastByte=Ttail;
				Open->Bhead=0;
				Irp->IoStatus.Information = TLastByte-Thead;
				Irp->IoStatus.Status = STATUS_SUCCESS;
				IoCompleteRequest(Irp, IO_NO_INCREMENT);
				return(STATUS_SUCCESS);
		}

		//scan the buffer to get the dimension to copy
		CpStart=Thead;
		while(TRUE){
			if(Thead==Ttail)break;
			if(Thead==TLastByte){
				PacketMoveMem(packp,CurrBuff+CpStart,Thead-CpStart,&(Open->Bhead));
				packp+=(Thead-CpStart);
				Open->Bhead=0;
				Thead=0;
				CpStart=0;
			}
			cplen=((struct bpf_hdr*)(CurrBuff+Thead))->bh_caplen+sizeof(struct bpf_hdr);
			if((i+cplen > Input_Buffer_Length))break; //no more space in the application buffer
			cplen=Packet_WORDALIGN(cplen);
			i+=cplen;
			Thead+=cplen;
		}

	//perform the copy
	PacketMoveMem(packp,CurrBuff+CpStart,Thead-CpStart,&(Open->Bhead));

	Open->Bhead=Thead;

	Irp->IoStatus.Information = i;
	Irp->IoStatus.Status = STATUS_SUCCESS;
    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    return(STATUS_SUCCESS);

}

//-------------------------------------------------------------------
NDIS_STATUS Packet_tap (IN NDIS_HANDLE ProtocolBindingContext,IN NDIS_HANDLE MacReceiveContext,
                        IN PVOID HeaderBuffer,IN UINT HeaderBufferSize,IN PVOID LookAheadBuffer,
                        IN UINT LookaheadBufferSize,IN UINT PacketSize)
{
    POPEN_INSTANCE      Open;
    PIO_STACK_LOCATION  IrpSp;
    PIRP                Irp;
    PLIST_ENTRY         PacketListEntry;
    PNDIS_PACKET        pPacket,pPacketb;
    ULONG               SizeToTransfer;
    NDIS_STATUS         Status;
    UINT                BytesTransfered;
    ULONG               BufferLength;
    PPACKET_RESERVED    Reserved;
    PMDL                pMdl,pMdlt;
	LARGE_INTEGER		CapTime;
	LARGE_INTEGER		TimeFreq;
	struct bpf_hdr		*header;

	KIRQL				OldIrql;
	PUCHAR				CurrBuff;
	UINT				Thead;
	UINT				Ttail;
	UINT				TLastByte;
	UINT				fres; //filter result
	UINT				maxbufspace;

    IF_LOUD(DbgPrint("Packet: ReceiveIndicate\n");)

	Open= (POPEN_INSTANCE)ProtocolBindingContext;

	Open->Received++;		//number of packets received by filter ++

	/*
	Check if the lookahead buffer follows the mac header.
	If the data follow the header (i.e. there is only a buffer) a normal bpf_filter() is
	executed on the packet.
	Otherwise if there are 2 separate buffers (this could be the case of LAN emulation or
	things like this) bpf_filter_with_2_buffers() is executed.
	*/

	if((UINT)LookAheadBuffer-(UINT)HeaderBuffer != HeaderBufferSize)
		fres=bpf_filter_with_2_buffers((struct bpf_insn*)(Open->bpfprogram),
									   HeaderBuffer,
									   LookAheadBuffer,
									   HeaderBufferSize,
									   PacketSize+HeaderBufferSize,
									   LookaheadBufferSize+HeaderBufferSize);
	else
		fres=bpf_filter((struct bpf_insn*)(Open->bpfprogram),
		                HeaderBuffer,
						PacketSize+HeaderBufferSize,
						LookaheadBufferSize+HeaderBufferSize);

	if(fres==0)return NDIS_STATUS_NOT_ACCEPTED;
	if( fres==-1)fres=PacketSize+HeaderBufferSize;	//if the filter returns -1 the whole packet must be accepted


    if (HeaderBufferSize > ETHERNET_HEADER_LENGTH)
    {
		return NDIS_STATUS_NOT_ACCEPTED;
    }

    //  See if there are any pending read that we can satisfy
    PacketListEntry=ExInterlockedRemoveHeadList(&Open->RcvList,&Open->RcvQSpinLock);
    if (PacketListEntry == NULL)
	/*no read requests. This means that the application is not performing a read in this 
	  moment: the packet must go in the buffer*/
    {

	if(Open->BufSize==0)return NDIS_STATUS_NOT_ACCEPTED;

	Thead=Open->Bhead;
	Ttail=Open->Btail;
	TLastByte=Open->BLastByte;

	maxbufspace=fres+sizeof(struct bpf_hdr);
	if(Ttail+maxbufspace>=Open->BufSize){
		if(Thead<=maxbufspace)
		{
			//the buffer is full: the packet is lost
			Open->Dropped++;
			return NDIS_STATUS_NOT_ACCEPTED;
		}
		else{
			Ttail=0;
		}
	}

	if((Ttail<Thead)&&(Ttail+maxbufspace>=Thead))
	{
			//the buffer is full: the packet is lost
			Open->Dropped++;
			return NDIS_STATUS_NOT_ACCEPTED;
	}
	CurrBuff=Open->Buffer+Ttail;

		pMdlt=IoAllocateMdl(CurrBuff,maxbufspace+sizeof(struct bpf_hdr),FALSE,FALSE,NULL);
		if (pMdlt == NULL)
		{
			//no memory: packet lost
		    IF_LOUD(DbgPrint("Packet: Read-Failed to allocate Mdl\n");)
			Open->Dropped++;
			return NDIS_STATUS_NOT_ACCEPTED;
		}
		//  Clear the next link in the new MDL
		pMdlt->Next=NULL;
		
		MmBuildMdlForNonPagedPool(pMdlt);

	    //  Allocate an MDL to map the portion of the buffer following the
	    //  header
	    pMdl=IoAllocateMdl(CurrBuff,maxbufspace+sizeof(struct bpf_hdr),FALSE,FALSE,NULL);
		if (pMdl == NULL)
		{
			//no memory: packet lost
			IF_LOUD(DbgPrint("Packet: Read-Failed to allocate Mdl\n");)
		    IoFreeMdl(pMdlt);
			Open->Dropped++;
			return NDIS_STATUS_NOT_ACCEPTED;
		}
		//  Build the mdl to point to the the portion of the buffer followin
		//  the header
		IoBuildPartialMdl(pMdlt,pMdl,((PUCHAR)MmGetMdlVirtualAddress(pMdlt))+HeaderBufferSize+sizeof(struct bpf_hdr),0);
		//  Clear the next link in the new MDL
		pMdl->Next=NULL;

		//allocate the packet from NDIS
		NdisAllocatePacket(&Status,&pPacketb,Open->PacketPool);
	    if (Status != NDIS_STATUS_SUCCESS)
		{
		    IF_LOUD(DbgPrint("Packet: Read- No free packets\n");)
		    IoFreeMdl(pMdl);
		    IoFreeMdl(pMdlt);
			Open->Dropped++;
			return NDIS_STATUS_NOT_ACCEPTED;
		}
		//link the buffer to the packet
		NdisChainBufferAtFront(pPacketb,pMdl);


		BufferLength=fres-HeaderBufferSize;
		//  Find out how much to transfer
		SizeToTransfer = (PacketSize < BufferLength) ? PacketSize : BufferLength;
		//  copy the ethernet header into the actual readbuffer
		NdisMoveMappedMemory((CurrBuff)+sizeof(struct bpf_hdr),HeaderBuffer,HeaderBufferSize+LookaheadBufferSize);
		//  Call the Mac to transfer the packet
		NdisTransferData(&Status,Open->AdapterHandle,MacReceiveContext,0,SizeToTransfer,pPacketb,&BytesTransfered);
		//store the lenght of the packet

		if (Status != NDIS_STATUS_PENDING)
		{
			//store the capture time
			CapTime=KeQueryPerformanceCounter(&TimeFreq); 
			//call the filter to see if the packet is valid
				if( fres>BytesTransfered+HeaderBufferSize )fres=BytesTransfered+HeaderBufferSize;
				//fill the bpf header for this packet
				CapTime.QuadPart+=Open->StartTime.QuadPart;
				header=(struct bpf_hdr*)CurrBuff;
				header->bh_tstamp.tv_usec=(long)((CapTime.QuadPart%TimeFreq.QuadPart*1000000)/TimeFreq.QuadPart);
				header->bh_tstamp.tv_sec=(long)(CapTime.QuadPart/TimeFreq.QuadPart);
				header->bh_caplen=fres;
				header->bh_datalen=PacketSize+HeaderBufferSize;
				header->bh_hdrlen=sizeof(struct bpf_hdr);
				
				//update the buffer	
				Ttail+=Packet_WORDALIGN(fres+sizeof(struct bpf_hdr));
				if(Ttail>Thead)TLastByte=Ttail;
				Open->Btail=Ttail;
				Open->BLastByte=TLastByte;
		}
		//Free the MDLs that we allocated
		IoFreeMdl(pMdl);
		IoFreeMdl(pMdlt);
		//recylcle the packet
		NdisReinitializePacket(pPacketb);
		//Put the packet on the free queue
		NdisFreePacket(pPacketb);

        return NDIS_STATUS_SUCCESS;
    }

	//we have a pending read to satisfy

    Reserved=CONTAINING_RECORD(PacketListEntry,PACKET_RESERVED,ListElement);
    pPacket=CONTAINING_RECORD(Reserved,NDIS_PACKET,ProtocolReserved);
    Irp=RESERVED(pPacket)->Irp;
    IrpSp = IoGetCurrentIrpStackLocation(Irp);
    //  This is the length of our partial MDL
    BufferLength=IrpSp->Parameters.Read.Length-HeaderBufferSize;
    //  Find out how much to transfer
    SizeToTransfer = (PacketSize < BufferLength) ? PacketSize : BufferLength;
    //  copy the ethernet header into the actual readbuffer
    NdisMoveMappedMemory((PUCHAR)MmGetSystemAddressForMdl(Irp->MdlAddress)+sizeof(struct bpf_hdr),HeaderBuffer,HeaderBufferSize);
    //  Call the Mac to transfer the packet
    NdisTransferData(&Status,Open->AdapterHandle,MacReceiveContext,0,SizeToTransfer,pPacket,&BytesTransfered);

    if (Status != NDIS_STATUS_PENDING)
    {
	//  If it didn't pend, finish the work

		//get the timestamp
		CapTime=KeQueryPerformanceCounter(&TimeFreq); 

		//call the filter to see if the packet is valid
			if( fres>BytesTransfered+HeaderBufferSize )fres=BytesTransfered+HeaderBufferSize;
			//fill the bpf header for this packet
			CapTime.QuadPart+=Open->StartTime.QuadPart;
			header=(struct bpf_hdr*)MmGetSystemAddressForMdl(Irp->MdlAddress);
			header->bh_tstamp.tv_usec=(long)((CapTime.QuadPart%TimeFreq.QuadPart*1000000)/TimeFreq.QuadPart);
			header->bh_tstamp.tv_sec=(long)(CapTime.QuadPart/TimeFreq.QuadPart);
			header->bh_caplen=fres;
				header->bh_datalen=PacketSize+HeaderBufferSize;
			header->bh_hdrlen=sizeof(struct bpf_hdr);
			//call the completion routine
			PacketTransferDataComplete(Open,pPacket,Status,fres);
    }
	
	return NDIS_STATUS_SUCCESS;

}

//-------------------------------------------------------------------
VOID PacketTransferDataComplete (IN NDIS_HANDLE ProtocolBindingContext,IN PNDIS_PACKET pPacket,
                                 IN NDIS_STATUS Status,IN UINT BytesTransfered)
{
    PIO_STACK_LOCATION   IrpSp;
    POPEN_INSTANCE       Open;
    PIRP                 Irp;
    PMDL                 pMdl;

    IF_LOUD(DbgPrint("Packet: TransferDataComplete\n");)
    Open= (POPEN_INSTANCE)ProtocolBindingContext;
    Irp=RESERVED(pPacket)->Irp;
    IrpSp = IoGetCurrentIrpStackLocation(Irp);
    pMdl=RESERVED(pPacket)->pMdl;
    //  Free the MDL that we allocated
    IoFreeMdl(pMdl);
    //  recylcle the packet
    NdisReinitializePacket(pPacket);
    //  Put the packet on the free queue
    NdisFreePacket(pPacket);
    Irp->IoStatus.Status = Status;
    Irp->IoStatus.Information = BytesTransfered;
    IoCompleteRequest(Irp, IO_NO_INCREMENT);
    return;
}

//-------------------------------------------------------------------
VOID PacketReceiveComplete(IN NDIS_HANDLE ProtocolBindingContext)
{
    return;
}

//-------------------------------------------------------------------

VOID
PacketCancelRoutine (
    IN PDEVICE_OBJECT   DeviceObject,
    IN PIRP             Irp
    )

/*++
Routine Description:
    The cancel routine. It will remove the IRP from the queue 
    and will complete it. The cancel spin lock is already acquired 
    when this routine is called.

Arguments:
    DeviceObject - pointer to the device object.
    Irp - pointer to the IRP to be cancelled.
--*/
{

    POPEN_INSTANCE      open; 
    KIRQL               oldIrql;
    PIRP                irpToComplete = NULL;
    PLIST_ENTRY         thisEntry, nextEntry, listHead;
    PIRP                pendingIrp;
    PNDIS_PACKET        myPacket;
    PPACKET_RESERVED    reserved;
    PMDL                mdl;
    PIO_STACK_LOCATION  irpSp;

    irpSp = IoGetCurrentIrpStackLocation(Irp);

    open=irpSp->FileObject->FsContext;

    //
    // Don't assume that the IRP being cancelled is in the queue.
    // Only complete the IRP if it IS in the queue.
    //
    //
    // Must acquire the local spinlock before releasing 
    // the global cancel spinlock
    //

    oldIrql = Irp->CancelIrql;
    
    KeAcquireSpinLockAtDpcLevel(&open->RcvQSpinLock);

    IoReleaseCancelSpinLock( KeGetCurrentIrql() );

    //
    // Remove the IRP from the queue
    //
    listHead = &open->RcvList;
    
    for(thisEntry = listHead->Flink; 
        thisEntry != listHead;
        thisEntry = thisEntry->Flink)
    {

        reserved=CONTAINING_RECORD(thisEntry,PACKET_RESERVED,ListElement);
        myPacket=CONTAINING_RECORD(reserved,NDIS_PACKET,ProtocolReserved);
        pendingIrp = RESERVED(myPacket)->Irp;

        if (pendingIrp == Irp) 
        {
                RemoveEntryList(thisEntry);
                irpToComplete = pendingIrp;
                break;
        }
      
    }
    
    //
    // Release the queue spinlock
    //

    KeReleaseSpinLock(&open->RcvQSpinLock, oldIrql);

    //
    // Complete the IRP with status canclled
    //
    
    if(irpToComplete) {
    
        mdl=RESERVED(myPacket)->pMdl;

        //
        //  Free the MDL that we allocated
        //
        IoFreeMdl(mdl);

        //
        //  recylcle the packet
        //
        NdisReinitializePacket(myPacket);

        //
        //  Put the packet on the free queue
        //
        NdisFreePacket(myPacket);

        irpToComplete->IoStatus.Status = STATUS_CANCELLED;
        irpToComplete->IoStatus.Information = 0;
        IoCompleteRequest(irpToComplete, IO_NO_INCREMENT);
    }

    return;
 
}

