/* Main driver file.

Copyright (C) 1999 Politecnico di Torino

This file is part of the NDIS Packet capture driver.

The GNU C Library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

The GNU C Library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with the GNU C Library; see the file COPYING.LIB.  If
not, write to the Free Software Foundation, Inc., 675 Mass Ave,
Cambridge, MA 02139, USA.  */

#include <basedef.h>
#include <vmm.h>
#include <ndis.h>
#include <vwin32.h>
#include "debug.h"
#include "packet.h"
#include "..\inc\ntddpack.h"
#pragma VxD_LOCKED_CODE_SEG
#pragma VxD_LOCKED_DATA_SEG

static POPEN_INSTANCE    Open;
static int nInstances=0;
NDIS_SPIN_LOCK nInstancesLock;

/*definition of the head of the open instances*/

PDEVICE_EXTENSION GlobalDeviceExtension = 0;

/************************************************************
    This routine initializes the Packet driver.
Arguments:
    DriverObject - Pointer to driver object created by system.
    RegistryPath - Pointer to the Unicode name of the registry path
        for this driver.
Return Value:
    The function value is the final status from the initialization operation.
************************************************************/
NTSTATUS
DriverEntry( IN PDRIVER_OBJECT	DriverObject,
				 IN PUNICODE_STRING	RegistryPath
	)
{
	NDIS_PROTOCOL_CHARACTERISTICS	ProtocolChar;
	NDIS_STRING	ProtoName = NDIS_STRING_CONST("VPACKET");
   	NDIS_HANDLE NdisProtocolHandle;
	NDIS_STATUS	Status;
	INIT_ENTER( "DriverEntry" );

	NdisAllocateMemory( (PVOID *)&GlobalDeviceExtension, sizeof( DEVICE_EXTENSION ), 0, -1 );
	if ( GlobalDeviceExtension != NULL )
	{
		NdisZeroMemory( (UCHAR*)GlobalDeviceExtension, sizeof(DEVICE_EXTENSION) );
		NdisZeroMemory( (UCHAR*)&ProtocolChar, sizeof(NDIS_PROTOCOL_CHARACTERISTICS) );
   		ProtocolChar.MajorNdisVersion            = 0x03;
		ProtocolChar.MinorNdisVersion            = 0x0A;
   		ProtocolChar.Reserved                    = 0;
		ProtocolChar.OpenAdapterCompleteHandler  = PacketOpenAdapterComplete;
   		ProtocolChar.CloseAdapterCompleteHandler = PacketUnbindAdapterComplete;
		ProtocolChar.SendCompleteHandler         = PacketSendComplete;
   		ProtocolChar.TransferDataCompleteHandler = PacketTransferDataComplete;
		ProtocolChar.ResetCompleteHandler        = PacketResetComplete;
   		ProtocolChar.RequestCompleteHandler      = PacketRequestComplete;
		ProtocolChar.ReceiveHandler              = Packet_tap;
   		ProtocolChar.ReceiveCompleteHandler      = PacketReceiveComplete;
		ProtocolChar.StatusHandler               = PacketStatus;
   		ProtocolChar.StatusCompleteHandler       = PacketStatusComplete;
   		ProtocolChar.BindAdapterHandler			 = PacketBindAdapter;
   		ProtocolChar.UnbindAdapterHandler        = PacketUnbindAdapter;
   		ProtocolChar.UnloadProtocolHandler       = PacketUnload;
		ProtocolChar.Name                        = ProtoName;
		NdisRegisterProtocol( &Status,
									 &GlobalDeviceExtension->NdisProtocolHandle,
									 &ProtocolChar,
									 sizeof(NDIS_PROTOCOL_CHARACTERISTICS) );
		if (Status != NDIS_STATUS_SUCCESS) 
   		{
			NdisFreeMemory( GlobalDeviceExtension, sizeof( DEVICE_EXTENSION ) ,  0 );
	   		IF_TRACE( "Failed to register protocol with NDIS" );
			INIT_LEAVE( "DriverEntry" );
			return Status;
   		}
		/*initializes the list of the open instances*/
		InitializeListHead( &GlobalDeviceExtension->OpenList );
		GlobalDeviceExtension->DriverObject = DriverObject;

		/*initializes lock on the instance's number*/
		NdisAllocateSpinLock( &(nInstancesLock) );

  		IF_TRACE( "protocol registered with NDIS!!! :-)" );
		INIT_LEAVE( "DriverEntry" );
		return Status;
	}
	IF_TRACE( "Memory Failure" );
	INIT_LEAVE( "DriverEntry" );
	return NDIS_STATUS_RESOURCES;
}

/************************************************************
Callback function called by NDIS when the protocol driver 
must be uninstalled 
INPUT:
OUTPUT:
************************************************************/
VOID NDIS_API PacketUnload()
{
	NDIS_STATUS		Status;
	TRACE_ENTER( "Unload" );
	if ( GlobalDeviceExtension )
	{
		NdisDeregisterProtocol( &Status, GlobalDeviceExtension->NdisProtocolHandle );
		if ( Status == NDIS_STATUS_SUCCESS )
		{
			NdisFreeMemory( GlobalDeviceExtension, sizeof( DEVICE_EXTENSION ) ,  0 );
			GlobalDeviceExtension = 0;
		}
	}
	TRACE_LEAVE( "Unload" );
	return;
}

/************************************************************
this function returns the descriptor of the adapter whose name
is passes as a parameter 
INPUT: Name of the adapter to open
OUTPUT:	instance of the driver
************************************************************/
POPEN_INSTANCE GetAdapterByName(BYTE *lpzAdapterName )
{
	DWORD                           dwBytes = 0;
      BYTE                            *lpzName;
      POPEN_INSTANCE                  pOpen;
	PWRAPPER_MAC_BLOCK	        pWMBlock;
	PNDIS_MAC_CHARACTERISTICS	  pNMChar;


      
	PLIST_ENTRY pHead = &(GlobalDeviceExtension->OpenList);
	PLIST_ENTRY pEntry;

	TRACE_ENTER( "GetAdapterByName" );
	IF_TRACE_MSG( " Passato %s", lpzAdapterName );	
	pOpen = 0;
	
	pEntry=pHead->Flink; 
	do {    
	pOpen = CONTAINING_RECORD( pEntry, OPEN_INSTANCE, ListElement );
 	pWMBlock = ((PWRAPPER_OPEN_BLOCK)(pOpen->AdapterHandle))->MacHandle;
	pNMChar  = &pWMBlock->MacCharacteristics;
	lpzName  = pNMChar->Name.Buffer;
	pEntry=pEntry->Flink;
	}while ((pEntry != pHead) && !(confronta(lpzName,lpzAdapterName))); 
 	  
	IF_TRACE_MSG( " Attuale %s", lpzName ); 
	if(confronta(lpzName,lpzAdapterName)==0) pOpen=NULL;
	
	TRACE_LEAVE( "GetAdapterByName" );
	
	return pOpen;
}

/************************************************************
This function evaluate the length of a string.
Useful to avoid the string library functions that are not 
defined at this level
************************************************************/
ULONG
strlen( BYTE *s )
{
	ULONG len = 0;
	while ( *s++ ) len++;
	return len;
}


/************************************************************
This function compares two strings
Useful to avoid the string library functions that are not 
defined at this level
************************************************************/
BYTE confronta(BYTE *s1,BYTE *s2)
{
	TRACE_ENTER( "confronta!" );	

	while (*s1 && *s2)
	{
		if (*s1!=*s2)  return (BYTE) 0;
					
		s1++;
		s2++;
			
	}

	if ((*s1==0) && (*s2==0)) return (BYTE) 1;	
	else return (BYTE) 0;
} 


/************************************************************
Return the names of all the MAC drivers on which the driver 
is attached
INPUT:	dwDDB e hDevice - parameters coming from the 
		DeviceIOControl procedure, not used here.
OUTPUT:	pDiocParms - structure containing the returned buffer
************************************************************/
DWORD
PacketGetMacNameList( DWORD  				dwDDB,
                      DWORD  				hDevice,
                      PDIOCPARAMETERS	pDiocParms ) 
{
        DWORD                                   dwBytes = 0;
        BYTE                                    *lpzName;
        ULONG                                   uLength;
        POPEN_INSTANCE                           pOpen;
	PWRAPPER_MAC_BLOCK			pWMBlock;
	PNDIS_MAC_CHARACTERISTICS	pNMChar;

	
	PLIST_ENTRY	pHead = &(GlobalDeviceExtension->OpenList);
	PLIST_ENTRY pEntry;
	TRACE_ENTER( "GetMacNameList" );
	for ( pEntry=pHead->Flink; pEntry != pHead; pEntry=pEntry->Flink )
	{
		pOpen = CONTAINING_RECORD( pEntry, OPEN_INSTANCE, ListElement );
		pWMBlock = ((PWRAPPER_OPEN_BLOCK)(pOpen->AdapterHandle))->MacHandle;
		pNMChar  = &pWMBlock->MacCharacteristics;
		lpzName  = pNMChar->Name.Buffer;
		uLength  = strlen( lpzName );
	
		IF_TRACE_MSG2( "     %s  %lu",  lpzName, uLength );
	
		if ( uLength < pDiocParms->cbOutBuffer - dwBytes - 1 )
		{
			strcat( (BYTE*)(pDiocParms->lpvOutBuffer), lpzName );
			strcat( (BYTE*)(pDiocParms->lpvOutBuffer), " " );
			dwBytes += (uLength + 1);
		}
		else
			break;
	}
	*(ULONG*)(pDiocParms->lpcbBytesReturned) = dwBytes;
	IF_TRACE_MSG( "     Bytes Returned: %lu", *(ULONG*)(pDiocParms->lpcbBytesReturned) );
	TRACE_LEAVE( "GetMacNameList" );
	return NDIS_STATUS_SUCCESS;
}


/************************************************************
This is the dispatch routine for create/open and close requests.
These requests complete successfully.
INPUT:	dwDDB e hDevice - parameters sent by the DeviceIOControl procedure
		dwService - requested service
		pDiocParms - structure containing the parameters of the call
OUTPUT:	the status of the operation
************************************************************/
 
DWORD _stdcall PacketIOControl( DWORD  			dwService,
                                DWORD  			dwDDB,
                                DWORD  			hDevice,
                                PDIOCPARAMETERS pDiocParms ) 
{
	PUCHAR				tpointer;
	int					*StatsBuf;
	PUCHAR				prog;
	ULONG				dim;
	NDIS_STATUS			Status;
	PPACKET_OID_DATA	reqbuff;
	
	
	TRACE_ENTER( "DeviceIoControl" );

	switch ( dwService )
	{

	case IOCTL_OPEN:	//open message


		//checks if there are other running instances
		//only one instance at the same time can run
		NdisAcquireSpinLock( &nInstancesLock );

		if(nInstances>=1){
			NdisReleaseSpinLock( &nInstancesLock );
			((BYTE*) pDiocParms->lpvInBuffer)[0]='\0';
			return NDIS_STATUS_FAILURE;
		}

		nInstances=1;

		NdisReleaseSpinLock( &nInstancesLock );


		Open = GetAdapterByName((BYTE*) pDiocParms->lpvInBuffer);
		
		if (Open==NULL){
			Open->BufSize=0;	//this is necessary because stops the buffering
			((BYTE*) pDiocParms->lpvInBuffer)[0]='\0';
			return NDIS_STATUS_FAILURE;
		}
	
		return PacketOpen(Open, dwDDB, hDevice, pDiocParms);
		
		break;


	case BIOCGSTATS: //fuction to obtain the capture stats

		StatsBuf=(int*)pDiocParms->lpvOutBuffer;
		StatsBuf[0]=Open->Received;
		StatsBuf[1]=Open->Dropped;
		*(DWORD *)(pDiocParms->lpcbBytesReturned) = 8;
		return NDIS_STATUS_SUCCESS;

	break;


	case BIOCSETF:  //fuction to set a new bpf filter

		/*free the previous buffer if it was present*/
		if(Open->bpfprogram!=NULL){
			NdisFreeMemory(Open->bpfprogram,Open->bpfprogramlen,-1);
			Open->bpfprogram=NULL; //NULL means accept all 
			Open->bpfprogramlen=0;
		}

		/*get the pointer to the new program*/
		prog=(PUCHAR)pDiocParms->lpvInBuffer;

		/*before accepting the program we must check that it's valid
		Otherwise, a bogus program could easily crash the system*/
		
		Open->bpfprogramlen=pDiocParms->cbInBuffer;
		if(bpf_validate((struct bpf_insn*)prog,Open->bpfprogramlen/sizeof(struct bpf_insn))==0)
		{
			Open->bpfprogramlen=0;
			Open->bpfprogram=NULL; 
			return NDIS_STATUS_FAILURE; // filter not accepted
		}

		/*allocate the memory to contain the new filter program*/
		if(NdisAllocateMemory(&Open->bpfprogram,Open->bpfprogramlen, 0, -1 )==NDIS_STATUS_FAILURE)
		{
			// no memory
			Open->bpfprogramlen=0;
			Open->bpfprogram=NULL; 
			return NDIS_STATUS_FAILURE;
		}

		/*copy the program in the new buffer*/
		NdisMoveMemory(Open->bpfprogram,prog,Open->bpfprogramlen);

		/*reset the buffer that could contain packets that don't match the filter*/
		Open->Bhead=0;
		Open->Btail=0;
		Open->BLastByte=0;

		*(DWORD *)(pDiocParms->lpcbBytesReturned) = Open->bpfprogramlen;

	break;


	case BIOCSETBUFFERSIZE:	//function to set the dimension of the buffer for the packets

		/*get the size to allocate*/
		dim=((PULONG)pDiocParms->lpvInBuffer)[0];
		/*free the old buffer*/
		NdisFreeMemory(Open->Buffer,Open->BufSize,-1);

		Open->Buffer=NULL;
		/*allocate the new buffer*/

		NdisAllocateMemory( (PVOID *)&tpointer,dim, 0, -1 );
		if (tpointer==NULL)
			{
		    // no memory
			Open->BufSize=0;
			return NDIS_STATUS_FAILURE;
			}

		Open->Buffer=tpointer;
		Open->Bhead=0;
		Open->Btail=0;
		Open->BLastByte=0;
		Open->BufSize=(UINT)dim;


		*(DWORD *)(pDiocParms->lpcbBytesReturned) = dim;
		
		break;


	case IOCTL_CLOSE:	//close message

		Status=PacketClose( Open, dwDDB, hDevice, pDiocParms );

		NdisAcquireSpinLock( &nInstancesLock );
		nInstances=0;
		NdisReleaseSpinLock( &nInstancesLock );

		return Status;

	break;

	
	case DIOC_CLOSEHANDLE:

		return NDIS_STATUS_SUCCESS;
		
		break;

	
	case IOCTL_PROTOCOL_RESET:

		PacketReset( &Status, Open );
	
		break;

	
	case IOCTL_PROTOCOL_SET_OID:
	case IOCTL_PROTOCOL_QUERY_OID:
	case IOCTL_PROTOCOL_STATISTICS:
	
		if(Open!=NULL)
		return PacketRequest( Open, dwService, dwDDB, hDevice, pDiocParms );
		else return NDIS_STATUS_SUCCESS;

	case IOCTL_PROTOCOL_READ:

		return PacketRead( Open, dwDDB, hDevice, pDiocParms );

	case IOCTL_PROTOCOL_WRITE:

		return PacketWrite( Open, dwDDB, hDevice, pDiocParms );

	case IOCTL_PROTOCOL_MACNAME:
		PacketGetMacNameList( dwDDB, hDevice, pDiocParms );
		break;
      
	default: 
		/*unknown function*/
		*(DWORD *)(pDiocParms->lpcbBytesReturned) = 0;
		break;
	}
   TRACE_LEAVE( "DeviceIoControl" );
   
   return NDIS_STATUS_SUCCESS;
}



/************************************************************
Function called by NDIS when there is something to communicate
to the upper level
************************************************************/
VOID
PacketStatus(
    IN NDIS_HANDLE   ProtocolBindingContext,
    IN NDIS_STATUS   Status,
    IN PVOID         StatusBuffer,
    IN UINT          StatusBufferSize
    )
{
   TRACE_ENTER( "Status Indication" );
   TRACE_LEAVE( "Status Indication" );
   return;
}

/************************************************************
Complete the previous call
************************************************************/
VOID NDIS_API
PacketStatusComplete(
    IN NDIS_HANDLE  ProtocolBindingContext
    )
{
   TRACE_ENTER( "StatusIndicationComplete" );
   TRACE_LEAVE( "StatusIndicationComplete" );
   return;
}

/************************************************************
Removes an element from a list.
Performs a check to see if the list is empty
************************************************************/
PLIST_ENTRY
PacketRemoveHeadList(
    IN PLIST_ENTRY pListHead
    )
{
	if ( !IsListEmpty( pListHead ) )
	{
		PLIST_ENTRY pLE = RemoveHeadList( pListHead );
		return pLE;
	}
	return NULL;
}
